
#include <e32std.h>

#include "EmuApple.h"

#define debug(x)

TApplePeripheral::TApplePeripheral(TAppleII *apple)
{
	this->apple = apple;
}

int TApplePeripheral::doIO(int address, int value)
{
	return apple->noise();
}

int TApplePeripheral::doHighIO(int address, int value)
{
	return apple->noise();
}

TAppleII::TAppleII()
	{
		sndlistener = NULL;
		//slots = new Peripheral[8];
		for (int slot=0; slot<8; slot++)
			slots[slot] = new TNullPeripheral(this);
		mem = new byte[0x17000];
		hardReset();
	}

TAppleII::~TAppleII()
	{
		for (int slot=0; slot<8; slot++)
		{
			delete slots[slot];
		}
		delete mem;
	}

void TAppleII::hardReset()
{
	hwstate.grswitch = hwstate.kbdlatch = hwstate.soundstate = 0;
	hwstate.auxRAMselected = false;
	hwstate.auxRAMbank = 1;
	hwstate.writeinhibit = true;
	computeBankOffsets();
	setRangeDirty(0, 0xffff, true);
	for (int i=0; i<0xc000; i++)
		writeMemory(i, 0);
	reset();
}

void TAppleII::setupMemory()
{
	// first fill range from 0xC100-0xC7FF with peripheral PROM
	for (int slot=1; slot<8; slot++)
	{
		for (int i=0; i<256; i++)
		{
			int addr = 0xC000 + slot*0x100 + i;
			mem[addr] = (byte)slots[slot]->doHighIO(addr, -1);
		}
	}
	// fill 0xd000-0xffff with LC RAM
	setupLCMemory();
}

void TAppleII::setupLCMemory()
{
	// fill 0xd000-0xffff with LC RAM
	Mem::Copy(mem+0xd000, mem+0xd000+bank2rdoffset, 0x1000);
	Mem::Copy(mem+0xe000, mem+0xe000+bank1rdoffset, 0x2000);
}

TApplePeripheral *TAppleII::getPeripheral(int index)
{
	return slots[index];
}

void TAppleII::setPeripheral(int index, TApplePeripheral *peripheral)
{
	delete slots[index];
	slots[index] = peripheral;
	setupMemory();
}

void TAppleII::setRangeDirty(int startAddress, int endAddress, boolean value)
{
	int p1 = startAddress>>DIRTY_SHIFT;
	int p2 = endAddress>>DIRTY_SHIFT;
	Mem::Fill(dirty+p1, p2-p1+1, value);
	/*
	for (int i=p1; i<p2; i++) {
		dirty[i] = value;
	}
	*/
}

int TAppleII::isRangeDirty(int startAddress, int endAddress)
{
	int p1 = startAddress>>DIRTY_SHIFT;
	int p2 = endAddress>>DIRTY_SHIFT;
	for (int i=p1; i<p2; i++) {
		if (dirty[i])
			return true;
	}
	return false;
}

int TAppleII::doDebug(int mask)
{
	return 1;
}

int TAppleII::noise()
{
	return mem[state.getClock() & 0xffff];
}

int TAppleII::doIO(int address, int value)
{
	int slot = (address >> 4) & 0x0f;
	switch (slot) {
		case 0:
			// kbd
			return hwstate.kbdlatch;
		case 1:
			// kbd strobe
			clearStrobe();
			break;
        case 3:
            // spkr
            hwstate.soundstate = hwstate.soundstate ^ 1;
			if (sndlistener)
				sndlistener->soundChanged(state.clock, hwstate.soundstate);
            break;
		case 5:
			if ((address & 0x0f) < 8)
			{
   				// graphics
   				if ((address & 1) != 0)
   					hwstate.grswitch |= 1 << ((address >> 1) & 0x07);
   				else
   					hwstate.grswitch &= ~(1 << ((address >> 1) & 0x07));
   			    if (doDebug(DBG_GR))
   			    {
       				debug("switch " + Integer.toString(address, 16) +
       					" grswitch = " + String.valueOf(grswitch));
       			}
       		} else {
       		    // annunciators
       		}
       		break;
	    case 6:
	        // tapein, joystick, buttons
	        switch (address & 7)
	        {
	            // buttons (off)
	            case 1: case 2: case 3:
                       return noise() & 0x7f;
                   // joystick
                   case 4: case 5:
                       return noise() | 0x80;
                   default:
                       return noise();
               }
           case 7:
               // joy reset
               if (address == 0xc070)
                   return noise() | 0x80;
		case 8:
			return doLanguageCardIO(address, value);
		case 9: case 10: case 11: case 12: case 13: case 14: case 15:
		    if (doDebug((slot-8)<<8))
		    {
           		int result = slots[slot-8]->doIO(address, value);
           		debug("slot " + Integer.toString(slot-8) + " @ " + Integer.toString(address, 16)
           		  + " = " + Integer.toString(result, 16));
           		return result;
           	} else
   				return slots[slot-8]->doIO(address, value);
		default:
			;
	}
       return noise();
}

int TAppleII::readMemory(int address)
{
	if (doDebug(DBG_RDMEM))
		debug("Read @ " + Integer.toString(address, 16));
		// see if it's from main memory (0x0000-0xbfff)
	if (address < HW_LO)
		return (mem[address]);
		// see if it came from the ROM/LC area (0xd000-0xffff)
	if (address >= ROM_LO) {
		if (address >= 0xe000)
			return (mem[address + bank1rdoffset]);
		else
			return (mem[address + bank2rdoffset]);
	}
	// it must be an I/O location (0xc000-0xcfff)
	if (address < HW_LO + 0x100)
		return doIO(address, -1);
	else
		return slots[(address >> 8) & 7]->doHighIO(address, -1);
}

void TAppleII::writeMemory(int address, int value)
{
	if (doDebug(DBG_WRMEM))
		debug("Write " + Integer.toString(value, 16) +
			" -> " + Integer.toString(address, 16));
		// see if it's from main memory (0x0000-0xbfff)
	if (address < HW_LO) {
		mem[address] = (byte)value;
		setDirty(address, true);
		return;
	}

	// see if it came from the ROM/LC area (0xd000-0xffff)
	if (address >= ROM_LO && !hwstate.writeinhibit) {
		// if reading AUX RAM is enabled, write it to main area
		if (hwstate.auxRAMselected)
			mem[address] = (byte)value;
		// always write it to backup area
		if (address >= 0xe000)
			mem[address + bank1wroffset] = (byte)value;
		else
			mem[address + bank2wroffset] = (byte)value;
		return;
	}

	// it must be an I/O location (0xc000-0xcfff)
	if (address < HW_LO + 0x100)
		doIO(address, value);
	else
		slots[(address >> 8) & 7]->doHighIO(address, value);
}

byte* TAppleII::getMemoryBase()
{
	return mem;
}

void TAppleII::clearStrobe()
{
	hwstate.kbdlatch &= 0x7f;
	if (doDebug(DBG_KBD))
		debug("Clear strobe");
}

void TAppleII::pressKey(int key)
{
	key = (key | 0x80) & 0xff;
	// since we're an Apple II+, we don't do lowercase
	if (key >= 0xe1 && key <= 0xfa)
	    key -= 0x20;
	hwstate.kbdlatch = key;
	if (doDebug(DBG_KBD))
		debug("Key pressed : " + String.valueOf(key));
}

int TAppleII::doLanguageCardIO(int address, int value)
{

	switch (address & 0x0f) {
	/*
	 * Select aux RAM bank 2, write protected.
	 */
	case 0x0: case 0x4:
		hwstate.auxRAMselected = true;
		hwstate.auxRAMbank = 2;
		hwstate.writeinhibit = true;
		break;
	/*
	 * Select ROM, write enable aux RAM bank 2.
	 */
	case 0x1: case 0x5:
		hwstate.auxRAMselected = false;
		hwstate.auxRAMbank = 2;
		hwstate.writeinhibit = false;
		break;
	/*
	 * Select ROM, write protect aux RAM (either bank).
	 */
	case 0x2: case 0x6:
	case 0xA: case 0xE:
		hwstate.auxRAMselected = false;
		hwstate.writeinhibit = true;
		break;
	/*
	 * Select aux RAM bank 2, write enabled.
	 */
	case 0x3: case 0x7:
		hwstate.auxRAMselected = true;
		hwstate.auxRAMbank = 2;
		hwstate.writeinhibit = false;
		break;
	/*
	 * Select aux RAM bank 1, write protected.
	 */
	case 0x8: case 0xC:
		hwstate.auxRAMselected = true;
		hwstate.auxRAMbank = 1;
		hwstate.writeinhibit = true;
		break;
	/*
	 * Select ROM, write enable aux RAM bank 1.
	 */
	case 0x9: case 0xD:
		hwstate.auxRAMselected = false;
		hwstate.auxRAMbank = 1;
		hwstate.writeinhibit = false;
		break;
	/*
	 * Select aux RAM bank 1, write enabled.
	 */
	case 0xB: case 0xF:
		hwstate.auxRAMselected = true;
		hwstate.auxRAMbank = 1;
		hwstate.writeinhibit = false;
	}

	// reset language card constants
	//
	int b1 = bank1rdoffset;
	int b2 = bank2rdoffset;
	computeBankOffsets();
	if (b1 != bank1rdoffset || b2 != bank2rdoffset)
	{
		setupLCMemory();
	}

	return noise();
}

void TAppleII::computeBankOffsets()
{
	if (hwstate.auxRAMselected) {
		bank1rdoffset = 0x3000;		// map 0xd000-0xffff -> 0x10000-0x12fff
		if (hwstate.auxRAMbank == 2)
			bank2rdoffset = 0x6000;	// map 0xd000-0xdfff -> 0x13000-0x13fff
		else
			bank2rdoffset = 0x3000;	// map 0xd000-0xdfff -> 0x10000-0x10fff
	} else {
		// map ROM
		bank1rdoffset = 0x7000;		// map 0xd000-0xffff -> 0x14000-0x16fff
		bank2rdoffset = 0x7000;		// map 0xd000-0xdfff -> 0x14000-0x14fff
	}
	if (!hwstate.writeinhibit) {
		bank1wroffset = 0x3000;		// map 0xd000-0xffff -> 0x10000-0x12fff
		if (hwstate.auxRAMbank == 2)
			bank2wroffset = 0x6000;	// map 0xd000-0xdfff -> 0x13000-0x13fff
		else
			bank2wroffset = 0x3000;	// map 0xd000-0xdfff -> 0x10000-0x10fff
	}
}

void TAppleII::execute(int cycles)
{
	//curmem = this->mem;
	T6502Emulation::execute(cycles);
}

